% demo_biGaussianized_calibration
%
% demonstrates bi-Gaussianized calibration
%
% available from: 
%   https://forensic-data-science.net/calibration-and-validation/#biGauss
%
% see: 
%   Morrison G.S. (2023). Bi-Gaussianized calibration of likelihood ratios. 
%       Law, Probability & Risk, 23, mgae004. https://doi.org/10.1093/lpr/mgae004 
% 
% version 2024-05-02a
% 
% Geoffrey Stewart Morrison 
% http://geoff-morrison.net/
%
%
% Toolboxes used:
%   curve_fitting_toolbox
%   distrib_computing_toolbox
%   statistics_toolbox
%
% Function "biGaussianized_calibration.m" is provided in the same folder as the present script
% 
% Additional needed functions are provided in the accompanying "functions" folder
% 
% Data are provided in the accompanying "data" folder
% 
% 
% tested on Matlab R2023b
%


%% setup
clear all;
close all;
clc;
addpath('./functions/');


%% options

% seed for random-number generator
rng(0);

% for coefficient values for Cllr_to_sigma2 function, use best fit or upper of 95% confidence bounds
use_best_fit = true;

% plot 95% confidence bounds for fitted regression line lnCllr v sigma2
plot_bounds = false;

% parameters for simulated scores (two Gaussians with same variance)
% already perfectly calibrated
sigma_true = 3;
mu_s_true = sigma_true^2 / 2;
mu_d_true = -mu_s_true;
% % same values as used in no_calibration_metric.m
% mu_d_true = 3;
% mu_s_true = 6;
% sigma_true = 1;

% parameters for simulated scores (Gaussian and Gumbel distributions with different variances)
mu_d_Gaus = 0;
sigma_d_Gaus = 3;
mu_s_Gumb = sigma_d_Gaus*3;
sigma_s_Gumb = sigma_d_Gaus/3;

% sample size (number of simulated scores)
num_s_sample = 100;
num_d_sample = (num_s_sample^2 - num_s_sample)/2;
num_sample = num_s_sample + num_d_sample;

% methods for biGaussianized_calibration
methods = ["biGauss_EER", "biGauss_LogReg", "biGauss_KDE", "LogReg"];

% indices for methods, and which biGauss variant to plot
[I_biG_EER, I_biG_LogReg, I_biG_KDE, I_LogReg, I_PAV, I_true] = deal(1,2,3,4,5,6);
I_biG_plot = I_biG_LogReg;

% kappa coefficient for regularized logistic regression
kappa = 0.01;

% compare estimates of sigma from LogReg, ERR, and PAV methods
compare_sigma_estimates = false;
num_samples_sigma_estimates = 1000;

% evaluate speaker scores
% WARNING: Running the cross-validated evaluation of speaker scores takes a long time.
evaluate_speakers = false;

% evaluate glass scores
% WARNING: Running the cross-validated evaluation of glass scores takes a very long time.
evaluate_glass = false;

% compare Cllr variability due to sampling variability
compare_Cllr_variability = false;
num_samples_Cllr_variability = 1000;

% plotting options
plot_line_width = 1;
plot_font_size = 12;


%% example plot of perfectly-calibrated bi-Gaussian system

sigma_example = 3;
LR_example = 10;
plot_biGaussian(sigma_example, LR_example, 10, 1);


%% Derive Cllr to sigma2 function using numerical integration

% for a perfectly calibrated system consiting of two Gaussians with the same variance
% mu_d = -sigma2/2, mu_s = sigma2/2

sigma2_x = (0 : 0.25 : 25)';
sigma_x = sqrt(sigma2_x);

sigma2_x_half = sigma2_x/2;
num_sigma_x = length(sigma_x);

sigma_label = 0:5;
sigma2_label = sigma_label.^2;

% numerical integration
fun = @(x,mu,sigma) normpdf(x,mu,sigma).*log(1+exp(-x)); % function based on same-source sum within Cllr equation

Cllr_y = NaN(num_sigma_x,1);
for I_sigma_x = 2:num_sigma_x
    Cllr_y(I_sigma_x) = integral(@(x) fun(x,sigma2_x_half(I_sigma_x),sigma_x(I_sigma_x)), -Inf,Inf, 'AbsTol',1e-12, 'RelTol',1e-2);
end
Cllr_y = Cllr_y/log(2);
Cllr_y(1) = 1; % at sigma = 0, Cllr = 1
ln_Cllr_y = log(Cllr_y);

% plot lnCllr v sigma2
figure;
plot(sigma2_x, ln_Cllr_y, 'or', 'MarkerSize', 6);
ylabel('ln(\itC\rm_{llr})', 'FontSize',plot_font_size);
xlabel('\sigma^2', 'FontSize',plot_font_size);
set(gca, 'YLim', [min(ln_Cllr_y) 0], 'XLim', [sigma2_x(1), sigma2_x(end)]);
grid on

% fit a regression of lnCllr on sigma2
%
% curveFitter(sigma2_x, ln_Cllr_y);
%
% y = a+b*exp(-c*x)
% constraint: at x=0, y=0 (at sigma2=0, all LR=1, therefore Cllr=1 and lnCllr=0)
% therefore: a=-b
% y = -b+b*exp(-c*x)
% y = b*(exp(-c*x)-1)
%
% inverse function (ln_Cllr_y to sigma2_x):
% x = -log((y/b) + 1) / c
%
% General model:
%      f(x) = b*(exp(-c*x)-1)
% Coefficients (with 95% confidence bounds):
%        b =       17.67  (16.95, 18.38)
%        c =    0.009334  (0.008924, 0.009744)
% 
% Goodness of fit:
%   SSE: 0.01131
%   R-square: 0.9999
%   Adjusted R-square: 0.9999
%   RMSE: 0.01069

[fitted_model, gof] = createFit(sigma2_x, ln_Cllr_y);
fitted_model_bounds = confint(fitted_model);

% coefficient values for Cllr_to_sigma2 function
if use_best_fit % use best fit
    b = fitted_model.b;
    c = fitted_model.c;
else % use upper of 95% confidence bounds
    b = fitted_model_bounds(2,1);
    c = fitted_model_bounds(2,2);
end

% plot fitted regression line
ln_Cllr_y_fit = fitted_model.b*(exp(-fitted_model.c*sigma2_x)-1);
hold on
plot(sigma2_x, ln_Cllr_y_fit, '-b', 'LineWidth',plot_line_width);
if plot_bounds
    ln_Cllr_y_fit_lower = fitted_model_bounds(1,1)*(exp(-fitted_model_bounds(1,2)*sigma2_x)-1);
    ln_Cllr_y_fit_upper = fitted_model_bounds(2,1)*(exp(-fitted_model_bounds(2,2)*sigma2_x)-1);
    plot(sigma2_x, ln_Cllr_y_fit_lower, '--b', 'LineWidth',plot_line_width);
    plot(sigma2_x, ln_Cllr_y_fit_upper, '--b', 'LineWidth',plot_line_width);
end

% plot sigma2 v Cllr
figure;
Cllr_y_fit = exp(ln_Cllr_y_fit);
plot(Cllr_y_fit, sigma2_x, '-b', 'LineWidth',plot_line_width);
if plot_bounds
    Cllr_y_fit_lower = exp(ln_Cllr_y_fit_lower);
    Cllr_y_fit_upper = exp(ln_Cllr_y_fit_upper);
    hold on
    plot(Cllr_y_fit_lower, sigma2_x, '--b', 'LineWidth',plot_line_width);
    plot(Cllr_y_fit_upper, sigma2_x, '--b', 'LineWidth',plot_line_width);
end
xlabel('\itC\rm_{llr}', 'FontSize',plot_font_size);
ylabel('\sigma^2', 'FontSize',plot_font_size);
ax=gca;
set(ax, 'XLim', [0 1], 'YLim', [sigma2_x(1), sigma2_x(end)]);
set(ax, 'XTick', 0:.1:1);
ax.YLabel.String = '\sigma^2';
ax.YLabel.Rotation = 0;
grid on


%% EER to sigma2

% plot sigma2 v EER

EER_step = 0.005;
EER_x = EER_step : EER_step : 0.5;

sigma2_y = EER_to_sigma2(EER_x);

figure;
plot(EER_x, sigma2_y, '-b', 'LineWidth',plot_line_width);

xlabel('\itE_=', 'FontSize',plot_font_size);
ylabel('\sigma^2', 'FontSize',plot_font_size);
ax=gca;
set(ax, 'XLim', [0 0.5], 'YLim', [sigma2_x(1), sigma2_x(end)]);
set(ax, 'XTick', 0:.05:.5);
ax.YLabel.String = '\sigma^2';
ax.YLabel.Rotation = 0;
grid on


%% compare methods for obtaining target sigma2

if compare_sigma_estimates

    df = num_s_sample;
    sigma2_estimates = NaN(num_samples_sigma_estimates,4);

    plot_EER = false;
    if plot_EER
        EER = NaN(num_samples_sigma_estimates,1);
    end

    % initialize parallel processing
    poolobj = gcp('nocreate');
    if isempty(poolobj)
        numcores = feature('numcores');
        poolobj = parpool(numcores-1);
    end
    % initiate random stream to replicate results when using parfor
    sc = parallel.pool.Constant(RandStream('threefry4x64_20'));
    % initiate waitbar
    h_wait = waitbar(0,'Comparing methods for obtaining \sigma');
    h_wait.UserData = [0 num_samples_sigma_estimates];
    par_DataQueue = parallel.pool.DataQueue;
    afterEach(par_DataQueue, @(varargin) increment_waitbar(h_wait));

    parfor I_sample = 1:num_samples_sigma_estimates
        % assign random stream to replicate results when using parfor
        stream = sc.Value;
        stream.Substream = I_sample;

        % generate Monte Carlo sample
        x_s_train = randn(stream, [1 num_s_sample]) * sigma_true + mu_s_true;
        x_d_train = randn(stream, [1 num_d_sample]) * sigma_true + mu_d_true;

        % EER
        x_train = [x_d_train,x_s_train];
        [~, ID_sorted] = sort(x_train);
    
        II_s_train = [zeros(1,num_d_sample), ones(1,num_s_sample)];
        II_s_train_sorted = II_s_train(ID_sorted);
        II_d_train_sorted = 1-II_s_train_sorted;
        error_rate_s_train_sorted = (cumsum(II_s_train_sorted)/num_s_sample); % false-alarm rate
        error_rate_d_train_sorted = 1-(cumsum(II_d_train_sorted)/num_d_sample); % miss rate
        
        I_eer = find(error_rate_d_train_sorted < error_rate_s_train_sorted, 1);
        if isempty(I_eer)
            FAR = 0.5/num_d_sample;
            MR = 0.5/num_s_sample;
        else
            FAR = max([min(error_rate_d_train_sorted(I_eer-1:I_eer)), 0.5/num_d_sample]);
            MR = max([min(error_rate_s_train_sorted(I_eer-1:I_eer)), 0.5/num_s_sample]);
        end
        EER_target = mean([FAR, MR]);
        
        sigma2_estimate_EER = EER_to_sigma2(EER_target);

        if plot_EER
            EER(I_sample) = EER_target;
        end
    
        % LogReg
        w = train_llr_fusion_regularized(x_s_train, x_d_train, 0.5, kappa, df);
        lnLR_d_logreg_train = lin_fusion(w, x_d_train);
        lnLR_s_logreg_train = lin_fusion(w, x_s_train);
        
        Cllr_LogReg_target = cllr(lnLR_s_logreg_train, lnLR_d_logreg_train);
        sigma2_estimate_LogReg = Cllr_to_sigma2(Cllr_LogReg_target);

        % KDE
        L_d_train = fitdist(x_d_train', 'Kernel');
        L_s_train = fitdist(x_s_train', 'Kernel');
        lnLR_KDE = log(pdf(L_s_train, x_train)) - log(pdf(L_d_train, x_train));

        Cllr_KDE_target = cllr(lnLR_KDE(num_d_sample+1:end), lnLR_KDE(1:num_d_sample));
        sigma2_estimate_KDE = Cllr_to_sigma2(Cllr_KDE_target);

        % PAV
        lnLR_pav = pav_transform(x_train, x_s_train, x_d_train);
    
        Cllr_PAV_target = cllr(lnLR_pav(num_d_sample+1:end), lnLR_pav(1:num_d_sample));
        sigma2_estimate_PAV = Cllr_to_sigma2(Cllr_PAV_target);

        % store values
        sigma2_estimates(I_sample,:) = [sigma2_estimate_EER, sigma2_estimate_LogReg, sigma2_estimate_KDE, sigma2_estimate_PAV]

        % update waitbar
        send(par_DataQueue, I_sample);
    
    end
    close(h_wait);

    sigma_estimates = sqrt(sigma2_estimates);
    
    % violin plots
    h_fig = figure;
    violin(sigma_estimates, 'xlabel',{'EER', 'LogReg', 'KDE', 'PAV'}, 'facecolor',[0.9 0.9 0.9], 'mc',[], 'plotlegend',false, 'same_area',true);
    ax=gca;
    ax.YLabel.String = '\sigma';
    ax.YLabel.Rotation = 0;
    hold on
    lim_x = get(ax,'XLim');
    plot(lim_x,[sigma_true;sigma_true], '-k', 'LineWidth',1)

    % set(ax,'YTick', [0:.2:2])

    % make figure wider, otherwise XTickLabels are hidden
    fig_position = h_fig.Position;
    fig_position(3) = fig_position(3)*1.1;
    h_fig.Position = fig_position;

    rms_sigma = rmse(sigma_estimates, sigma_true);
    
    fprintf('\n\nRMS error for sigma_target according to method\n')
    fprintf('EER\tLogReg\tKDE\tPAV\n')
    fprintf('%0.3f\t%0.3f\t%0.3f\t%0.3f\n', ...
        rms_sigma)

    if plot_EER
        figure;
        violin(EER);
    end

end


%% calibrate Monte Carlo simuluted data (two Gaussians with same variance)

% training sample
x_s_train = (random('Normal', mu_s_true, sigma_true, [num_s_sample 1]))';
x_d_train = (random('Normal', mu_d_true, sigma_true, [num_d_sample 1]))';
x_train = [x_s_train,x_d_train];
II_train = [true(1,num_s_sample), false(1,num_d_sample)];
[x_train_sorted, ID_train_sorted] = sort(x_train);
II_train_sorted = II_train(ID_train_sorted);

% test sample
x_s_test = (random('Normal', mu_s_true, sigma_true, [num_s_sample 1]))';
x_d_test = (random('Normal', mu_d_true, sigma_true, [num_d_sample 1]))';
x_test = [x_s_test,x_d_test];
II_s_test = [true(1,num_s_sample), false(1,num_d_sample)];

% coefficients for true LDF
% y = a + b*x;
b_true = (mu_s_true - mu_d_true) / sigma_true^2;
a_true = -b_true*(mu_s_true + mu_d_true)/2;

% bi-Gaussianized calibration
Cllr_to_sigma2_coefs = [b c];
logreg_regularization_coefs = [kappa, num_s_sample];

[lnLR_test, sigma2_target, lnLR_s_train, lnLR_d_train] = ...
biGaussianized_calibration ...
    (x_test, x_s_train, x_d_train, methods, ...
     Cllr_to_sigma2_coefs, logreg_regularization_coefs, I_biG_plot);

% xlim([-16 16])

% PAV calibration
lnLR_s_train(:,I_PAV) = pav_transform(x_s_train, x_s_train, x_d_train);
lnLR_d_train(:,I_PAV) = pav_transform(x_d_train, x_s_train, x_d_train);
lnLR_test(:,I_PAV) = pav_transform(x_test, x_s_train, x_d_train);

% LDF calibration
lnLR_s_train(:,I_true) = (log(normpdf(x_s_train, mu_s_true, sigma_true)) - log(normpdf(x_s_train, mu_d_true, sigma_true)))';
lnLR_d_train(:,I_true) = (log(normpdf(x_d_train, mu_s_true, sigma_true)) - log(normpdf(x_d_train, mu_d_true, sigma_true)))';
lnLR_test(:,I_true) = (log(normpdf(x_test, mu_s_true, sigma_true)) - log(normpdf(x_test, mu_d_true, sigma_true)))';

% target sigma
sigma_target = sqrt(sigma2_target);

fprintf('\n\nsimuluted data (two Gaussians with same variance)\n')
fprintf('\nsigma\ntrue\tbiG_EER\tbiG_LogReg\tbiG_KDE\n')
fprintf('%0.2f\t%0.2f\t%0.2f\t%0.2f\n', ...
    [sigma_true, sigma_target])

% Cllr
lnLR_s_test = lnLR_test(II_s_test,:);
lnLR_d_test = lnLR_test(~II_s_test,:);

Cllr_test = cllr(lnLR_s_test, lnLR_d_test);
Cllr_train = cllr(lnLR_s_train, lnLR_d_train);

fprintf('\nCllr\t(top row train, bottom row test)')
fprintf('\nbiG_EER\tbiG_LogReg\tbiG_KDE\tLogReg\tPAV\ttrue\n')
fprintf('%0.3f\t%0.3f\t%0.3f\t%0.3f\t%0.3f\t%0.3f\n', ...
    [Cllr_train; Cllr_test]')

% plot mapping functions
figure;
minmax_x_train = [x_train_sorted(1); x_train_sorted(end)]';
plot(minmax_x_train, [0, 0], '-k')
hold on
% true
lnLR_true_minmax = a_true + b_true*minmax_x_train;
h1 = plot(minmax_x_train, lnLR_true_minmax, '-g', 'LineWidth',plot_line_width);
% PAV
lnLR_pav_sorted = sort([lnLR_d_train(:,I_PAV); lnLR_s_train(:,I_PAV)])';
h2 = plot(x_train_sorted, lnLR_pav_sorted, '-.m', 'LineWidth',plot_line_width);
% logreg
minmax_logreg_train = minmax([lnLR_s_train(:,I_LogReg); lnLR_d_train(:,I_LogReg)]');
h3 = plot(minmax_x_train, minmax_logreg_train, '--r', 'LineWidth',plot_line_width);
% biGauss
lnLR_biGaus_sorted = sort([lnLR_s_train(:,I_biG_plot); lnLR_d_train(:,I_biG_plot)])';
h4 = plot(x_train_sorted, lnLR_biGaus_sorted, ':b', 'LineWidth',plot_line_width);
xlabel('Score', 'FontSize',plot_font_size);
ylabel('ln(\Lambda)', 'FontSize',plot_font_size);
axis tight
grid on
title(['\sigma true = ', num2str(sigma_true, '%0.2f'), ', \sigma target = ', num2str(sigma_target(I_biG_plot), '%0.2f')]);
legend([h1,h2,h3,h4],{'true', 'PAV', 'LogReg', 'biGauss'}, 'Location','northwest')

% xlim([-16 16])


%% calibrate simulated data (Gaussian and Gumbel distributions with different variances)

% simulate data
x_d_Gaus_train = sort((random('Normal', mu_d_Gaus, sigma_d_Gaus, [num_d_sample 1])))';
x_s_Gumb_train = sort(-(random('Generalized Extreme Value', 0, sigma_s_Gumb, -mu_s_Gumb, [num_s_sample 1])))';

x_d_Gaus_test = sort((random('Normal', mu_d_Gaus, sigma_d_Gaus, [num_d_sample 1])))';
x_s_Gumb_test = sort(-(random('Generalized Extreme Value', 0, sigma_s_Gumb, -mu_s_Gumb, [num_s_sample 1])))';
x_GausGumb_test = [x_s_Gumb_test,x_d_Gaus_test];
II_s_GausGumb_test = [true(1,num_s_sample), false(1,num_d_sample)];

x_GausGumb_train = [x_s_Gumb_train,x_d_Gaus_train];
[x_sample_sorted_GausGumb, ID_GausGumb_train_sorted] = sort(x_GausGumb_train);
II_GausGum_train_sorted = II_train(ID_GausGumb_train_sorted);

% plot true distribution
minmax_x_GausGumb_train = minmax(x_GausGumb_train);
step_x_GausGumb = (minmax_x_GausGumb_train(2)-minmax_x_GausGumb_train(1))/399;
x_plot_GausGumb_train = minmax_x_GausGumb_train(1) : step_x_GausGumb : minmax_x_GausGumb_train(2);

y_d_Gaus_train = pdf('Normal', x_plot_GausGumb_train, mu_d_Gaus, sigma_d_Gaus);
y_s_Gumb_train = pdf('Generalized Extreme Value', mu_s_Gumb-x_plot_GausGumb_train, 0, sigma_s_Gumb, 0);

figure;
plot(x_plot_GausGumb_train, y_d_Gaus_train, '-r', 'LineWidth',plot_line_width);
hold on
plot(x_plot_GausGumb_train, y_s_Gumb_train, '-b', 'LineWidth',plot_line_width);
xlim(minmax_x_GausGumb_train);
xlabel('score', 'FontSize',plot_font_size);
ylabel('probability density', 'FontSize',plot_font_size);
grid on

% biGauss calibration
[lnLR_GausGumb_test, sigma2_target_GausGumb, lnLR_GausGumb_s_train, lnLR_GausGumb_d_train] = ...
biGaussianized_calibration ...
    (x_GausGumb_test, x_s_Gumb_train, x_d_Gaus_train, methods, ...
     Cllr_to_sigma2_coefs, logreg_regularization_coefs, I_biG_plot);

% xlim([-16 16])

% PAV calibration
lnLR_GausGumb_s_train(:,I_PAV) = pav_transform(x_s_Gumb_train, x_s_Gumb_train, x_d_Gaus_train);
lnLR_GausGumb_d_train(:,I_PAV) = pav_transform(x_d_Gaus_train, x_s_Gumb_train, x_d_Gaus_train);
lnLR_GausGumb_test(:,I_PAV) = pav_transform(x_GausGumb_test, x_s_Gumb_train, x_d_Gaus_train);

% true lnLRs based on parameters
lnLR_GausGumb_s_train(:,I_true) = (log(pdf('Generalized Extreme Value', mu_s_Gumb-x_s_Gumb_train, 0, sigma_s_Gumb, 0)) - log(normpdf(x_s_Gumb_train, mu_d_Gaus, sigma_d_Gaus)))';
lnLR_GausGumb_d_train(:,I_true) = (log(pdf('Generalized Extreme Value', mu_s_Gumb-x_d_Gaus_train, 0, sigma_s_Gumb, 0)) - log(normpdf(x_d_Gaus_train, mu_d_Gaus, sigma_d_Gaus)))';
lnLR_GausGumb_test(:,I_true) = (log(pdf('Generalized Extreme Value', mu_s_Gumb-x_GausGumb_test, 0, sigma_s_Gumb, 0)) - log(normpdf(x_GausGumb_test, mu_d_Gaus, sigma_d_Gaus)))';

% target sigma
sigma_target_GausGumb = sqrt(sigma2_target_GausGumb);

fprintf('\n\nsimuluted data (Gaussian and Gumbel distributions with different variances)\n')
fprintf('\nsigma\nbiG_EER\tbiG_LogReg\tbiG_KDE\n')
fprintf('%0.2f\t%0.2f\t%0.2f\n', ...
    sigma_target_GausGumb)

% Cllr
lnLR_GausGumb_s_test = lnLR_GausGumb_test(II_s_GausGumb_test,:);
lnLR_GausGumb_d_test = lnLR_GausGumb_test(~II_s_GausGumb_test,:);

Cllr_GausGumb_test = cllr(lnLR_GausGumb_s_test, lnLR_GausGumb_d_test);
Cllr_GausGumb_train = cllr(lnLR_GausGumb_s_train, lnLR_GausGumb_d_train);

fprintf('\nCllr\t(top row train, bottom row test)')
fprintf('\nbiG_EER\tbiG_LogReg\tbiG_KDE\tLogReg\tPAV\ttrue\n')
fprintf('%0.3f\t%0.3f\t%0.3f\t%0.3f\t%0.3f\t%0.3f\n', ...
    [Cllr_GausGumb_train; Cllr_GausGumb_test]')

% plot mapping functions
figure;
plot(minmax_x_GausGumb_train', [0; 0], '-k')
hold on
% true calibration based on parameter values
lnLR_true_GausGumb = [lnLR_GausGumb_s_train(:,I_true); lnLR_GausGumb_d_train(:,I_true)]';
lnLR_true_sorted_GausGumb = lnLR_true_GausGumb(ID_GausGumb_train_sorted);
h1 = plot(x_sample_sorted_GausGumb, lnLR_true_sorted_GausGumb, '-g', 'LineWidth',plot_line_width);
% PAV
lnLR_pav_GausGumb = [lnLR_GausGumb_s_train(:,I_PAV); lnLR_GausGumb_d_train(:,I_PAV)]';
lnLR_pav_GausGumb_sorted = lnLR_pav_GausGumb(ID_GausGumb_train_sorted);
h2 = plot(x_sample_sorted_GausGumb, lnLR_pav_GausGumb_sorted, '-.m', 'LineWidth',plot_line_width);
% logreg
lnLR_minmax_logreg_GausGumb = minmax([lnLR_GausGumb_s_train(:,I_LogReg); lnLR_GausGumb_d_train(:,I_LogReg)]');
h3 = plot(minmax_x_GausGumb_train, lnLR_minmax_logreg_GausGumb, '--r', 'LineWidth',plot_line_width);
% biGaus
lnLR_biGaus_GausGumb_train_sorted = sort([lnLR_GausGumb_s_train(:,I_biG_plot); lnLR_GausGumb_d_train(:,I_biG_plot)]);
h4 = plot(x_sample_sorted_GausGumb, lnLR_biGaus_GausGumb_train_sorted, ':b', 'LineWidth',plot_line_width);
xlabel('Score', 'FontSize',plot_font_size);
ylabel('ln(\Lambda)', 'FontSize',plot_font_size);
axis tight
grid on
title(['\sigma target = ', num2str(sigma_target_GausGumb(I_biG_plot), '%0.2f')]);
legend([h1,h2,h3,h4],{'true', 'PAV', 'LogReg', 'biGauss'}, 'Location','northwest')

% plot pdfs of lnLRs of training data
minmax_lnLR_GausGumb_train = [lnLR_biGaus_GausGumb_train_sorted(1); lnLR_biGaus_GausGumb_train_sorted(end)];
step_lnLR_GausGumb = (minmax_lnLR_GausGumb_train(2)-minmax_lnLR_GausGumb_train(1))/399;
lnLR_plot_GausGumb_train = minmax_lnLR_GausGumb_train(1) : step_lnLR_GausGumb : minmax_lnLR_GausGumb_train(2);

half_sigma2_target_GausGumb = sigma2_target_GausGumb(I_biG_plot)/2;

y_d_true_GausGumb_train = pdf('Normal', lnLR_plot_GausGumb_train, -half_sigma2_target_GausGumb, sigma_target_GausGumb(I_biG_plot));
y_s_true_GausGumb_train = pdf('Normal', lnLR_plot_GausGumb_train, half_sigma2_target_GausGumb, sigma_target_GausGumb(I_biG_plot));

kde_lnLR_d_biGaus_GausGumb_train = fitdist(lnLR_GausGumb_d_train(:,I_biG_plot), 'Kernel');
kde_lnLR_s_biGaus_GausGumb_train = fitdist(lnLR_GausGumb_s_train(:,I_biG_plot), 'Kernel');

y_d_biGaus_GausGumb_train = pdf(kde_lnLR_d_biGaus_GausGumb_train, lnLR_plot_GausGumb_train);
y_s_biGaus_GausGumb_train = pdf(kde_lnLR_s_biGaus_GausGumb_train, lnLR_plot_GausGumb_train);

kde_lnLR_d_logreg_GausGumb_train = fitdist(lnLR_GausGumb_d_train(:,I_LogReg), 'Kernel');
kde_lnLR_s_logreg_GausGumb_train = fitdist(lnLR_GausGumb_s_train(:,I_LogReg), 'Kernel');

y_d_plot_logreg_GausGumb_train = pdf(kde_lnLR_d_logreg_GausGumb_train, lnLR_plot_GausGumb_train);
y_s_plot_logreg_GausGumb_train = pdf(kde_lnLR_s_logreg_GausGumb_train, lnLR_plot_GausGumb_train);

figure;
h1 = plot(lnLR_plot_GausGumb_train, y_d_true_GausGumb_train, '-r', 'LineWidth',plot_line_width);
hold on
plot(lnLR_plot_GausGumb_train, y_s_true_GausGumb_train, '-b', 'LineWidth',plot_line_width);
h2 = plot(lnLR_plot_GausGumb_train, y_d_plot_logreg_GausGumb_train, '--r', 'LineWidth',plot_line_width);
plot(lnLR_plot_GausGumb_train, y_s_plot_logreg_GausGumb_train, '--b', 'LineWidth',plot_line_width);
h3 = plot(lnLR_plot_GausGumb_train, y_d_biGaus_GausGumb_train, ':r', 'LineWidth',plot_line_width);
plot(lnLR_plot_GausGumb_train, y_s_biGaus_GausGumb_train, ':b', 'LineWidth',plot_line_width);
xlim(minmax_lnLR_GausGumb_train)
lim_y = ylim;
h4 = plot([0 0], lim_y, '-k', 'LineWidth',plot_line_width/2);
ylim(lim_y);
uistack(h4,'bottom')
grid on
xlabel('ln(\Lambda)', 'FontSize',plot_font_size);
ylabel('probability density', 'FontSize',plot_font_size);
legend([h1,h2,h3],{'target', 'LogReg', 'biGauss'}, 'Location','northwest')
title('train')

% plot pdfs of lnLRs of test data
minmax_x_GausGumb_test = minmax([lnLR_GausGumb_d_test(:,I_biG_plot); lnLR_GausGumb_s_test(:,I_biG_plot)]');
step_x_GausGumb = (minmax_x_GausGumb_test(2)-minmax_x_GausGumb_test(1))/399;
x_GausGumb_test = minmax_x_GausGumb_test(1) : step_x_GausGumb : minmax_x_GausGumb_test(2);

y_d_target_GausGumb_test = pdf('Normal', x_GausGumb_test, -half_sigma2_target_GausGumb, sigma_target_GausGumb(I_biG_plot));
y_s_target_GausGumb_test = pdf('Normal', x_GausGumb_test, half_sigma2_target_GausGumb, sigma_target_GausGumb(I_biG_plot));

kde_lnLR_d_biGaus_GausGumb_test = fitdist(lnLR_GausGumb_d_test(:,I_biG_plot), 'Kernel');
kde_lnLR_s_biGaus_GausGumb_test = fitdist(lnLR_GausGumb_s_test(:,I_biG_plot), 'Kernel');

y_d_biGaus_GausGumb_test = pdf(kde_lnLR_d_biGaus_GausGumb_test, x_GausGumb_test);
y_s_biGaus_GausGumb_test = pdf(kde_lnLR_s_biGaus_GausGumb_test, x_GausGumb_test);

kde_lnLR_d_logreg_GausGumb_test = fitdist(lnLR_GausGumb_d_test(:,I_LogReg), 'Kernel');
kde_lnLR_s_logreg_GausGumb_test = fitdist(lnLR_GausGumb_s_test(:,I_LogReg), 'Kernel');

y_d_logreg_GausGumb_test = pdf(kde_lnLR_d_logreg_GausGumb_test, x_GausGumb_test);
y_s_logreg_GausGumb_test = pdf(kde_lnLR_s_logreg_GausGumb_test, x_GausGumb_test);

figure;
h1 = plot(x_GausGumb_test, y_d_target_GausGumb_test, '-r', 'LineWidth',plot_line_width);
hold on
plot(x_GausGumb_test, y_s_target_GausGumb_test, '-b', 'LineWidth',plot_line_width);
h2 = plot(x_GausGumb_test, y_d_logreg_GausGumb_test, '--r', 'LineWidth',plot_line_width);
plot(x_GausGumb_test, y_s_logreg_GausGumb_test, '--b', 'LineWidth',plot_line_width);
h3 = plot(x_GausGumb_test, y_d_biGaus_GausGumb_test, ':r', 'LineWidth',plot_line_width);
plot(x_GausGumb_test, y_s_biGaus_GausGumb_test, ':b', 'LineWidth',plot_line_width);
xlim(minmax_x_GausGumb_test)
% lim_y = ylim; % uncomment to make same height as previous figure
h4 = plot([0 0], lim_y, '-k', 'LineWidth',plot_line_width/2);
ylim(lim_y);
uistack(h4,'bottom')
grid on
xlabel('ln(\Lambda)', 'FontSize',plot_font_size);
ylabel('probability density', 'FontSize',plot_font_size);
legend([h1,h2,h3],{'target', 'LogReg', 'biGauss'}, 'Location','northwest')
title('test')

% % Tippett plot – using "plot_tippett" version 2023-07-01
% cum_prop_s_target = (1:num_s_sample)/(num_s_sample+1);
% lnLR_s_target_GausGumb = icdf('Normal', cum_prop_s_target, half_sigma2_target_GausGumb, sigma_target_GausGumb(I_biG_plot));
% cum_prop_d_target = (1:num_d_sample)/(num_d_sample+1);
% lnLR_d_target_GausGumb = icdf('Normal', cum_prop_d_target, -half_sigma2_target_GausGumb, sigma_target_GausGumb(I_biG_plot));
% 
% plot_tippett(exp(lnLR_s_target_GausGumb), [], exp(lnLR_d_target_GausGumb), [], [], true, '-', true, [], [], true);
% plot_tippett(exp(lnLR_GausGumb_s_test(:,I_LogReg)), [], exp(lnLR_GausGumb_d_test(:,I_LogReg)), [], [], false, '--', false, [], [], true);
% plot_tippett(exp(lnLR_GausGumb_s_test(:,I_biG_plot)), [], exp(lnLR_GausGumb_d_test(:,I_biG_plot)), [], [], false, ':', false, [], [], true);
% box on
% title('test')
% xlim([-8 8]) % may need to change this if parameter values for simulated scores are changed
% 
% h_cf = gcf;
% h1 = findobj(h_cf, 'Color','red', 'LineStyle','-');
% h2 = findobj(h_cf, 'Color','red', 'LineStyle','--');
% h3 = findobj(h_cf, 'Color','red', 'LineStyle',':');
% legend([h1,h2, h3],{'target', 'LogReg', 'biGauss'}, 'Location','SouthWest')

% Tippett plot – using "plot_tippett" version 2024-04-02
plot_tippett(exp(lnLR_GausGumb_s_test(:,I_LogReg)), [], exp(lnLR_GausGumb_d_test(:,I_LogReg)), [], [], true, '-', true, [], [], true, sqrt(sigma2_target_GausGumb(I_biG_plot)));
plot_tippett(exp(lnLR_GausGumb_s_test(:,I_LogReg)), [], exp(lnLR_GausGumb_d_test(:,I_LogReg)), [], [], false, '--', false, [], [], true);
plot_tippett(exp(lnLR_GausGumb_s_test(:,I_biG_plot)), [], exp(lnLR_GausGumb_d_test(:,I_biG_plot)), [], [], false, ':', false, [], [], true);
box on
title('test')
xlim([-8 8]) % may need to change this if parameter values for simulated scores are changed

h_cf = gcf;
h1 = findobj(h_cf, 'Color','red', 'LineStyle','-');
h2 = findobj(h_cf, 'Color','red', 'LineStyle','--');
h3 = findobj(h_cf, 'Color','red', 'LineStyle',':');
legend([h1,h2, h3],{'target', 'LogReg', 'biGauss'}, 'Location','SouthWest')


%% Real data: speakers

% raw scores output by E3FS3 in response to forensic_eval_01 test data
% see:
%   Weber P., Enzinger E., Labrador B., Lozano-Díez A., Ramos D., González-Rodríguez J., Morrison G.S. (2022). 
%       Validation of the alpha version of the E3 Forensic Speech Science System (E3FS3) core software tools. 
%       Forensic Science International: Synergy, 4, 100223. 
%       https://doi.org/10.1016/j.fsisyn.2022.100223
% 
% these scores were prodced by a later version of E3FS3 with slightly improved performance compared the version described in Weber et al (2022)

score_file_speakers = 'speaker_scores.csv';
data_speakers = readtable(['./data/', score_file_speakers]);

scores_speakers = data_speakers.score * 0.025; % rescale scores so that they plot in approximately the same range as the target biGaussian distribution
num_scores_speakers = length(scores_speakers);

indices_speakers_k_cell = data_speakers.recid1;
indices_speakers_q_cell = data_speakers.recid2;

indices_speakers_k = cellfun(@(str) uint16(str2double(str(1:4))), indices_speakers_k_cell);
indices_speakers_q = cellfun(@(str) uint16(str2double(str(1:4))), indices_speakers_q_cell);
indices_speakers = [indices_speakers_k, indices_speakers_q];

II_s_speakers = indices_speakers(:,1) == indices_speakers(:,2);
II_d_speakers = ~II_s_speakers;

scores_s_speakers = scores_speakers(II_s_speakers);
scores_d_speakers = scores_speakers(II_d_speakers);

indices_s_speakers = indices_speakers(II_s_speakers,:);
indices_d_speakers = indices_speakers(II_d_speakers,:);

unique_indices_speakers = unique(indices_s_speakers(:,1));
num_speakers = length(unique_indices_speakers);

% plot score distributions
minmax_scores_speakers = minmax(scores_speakers');
step_scores_speakers = (minmax_scores_speakers(2) - minmax_scores_speakers(1))/399;
x_scores_speakers = minmax_scores_speakers(1) : step_scores_speakers : minmax_scores_speakers(2);

kde_scores_s_speakers = fitdist(scores_s_speakers, 'Kernel');
kde_scores_d_speakers = fitdist(scores_d_speakers, 'Kernel');

y_d_scores_speakers = pdf(kde_scores_d_speakers, x_scores_speakers);
y_s_scores_speakers = pdf(kde_scores_s_speakers, x_scores_speakers);

figure;
plot(x_scores_speakers, y_d_scores_speakers, '-r', 'LineWidth',plot_line_width);
hold on
plot(x_scores_speakers, y_s_scores_speakers, '-b', 'LineWidth',plot_line_width);
xlim(minmax_scores_speakers)
grid on
xlabel('scores', 'FontSize',plot_font_size);
ylabel('probability density', 'FontSize',plot_font_size);
title('speakers: scores');

% run bi-Gaussian calibration (LogReg variant) without cross-validation in order to plot cdfs 
biGaussianized_calibration(scores_speakers, scores_s_speakers, scores_d_speakers, "biGauss_LogReg", Cllr_to_sigma2_coefs, [kappa, num_speakers], 1);
xlim(minmax_scores_speakers)

if evaluate_speakers
    fprintf('\n\nWARNING: Running the cross-validated evaluation of speaker scores takes a long time.\n')

    % leave-one-source-out leave-two-sources-out cross validation 

    % build cross-validation indices in preparation for parallel processing
    num_methods = 5;
    num_biG_methods = 3;
    num_speaker_pairs = (num_speakers^2 + num_speakers) / 2;

    II_cross_val = cell(num_speaker_pairs,2);
    II_pair_s_speakers = false(num_speaker_pairs,1);

    count_speaker_pairs = 0;
    h_wait = waitbar(count_speaker_pairs,'Building cross-validation indices');
    for I_1 = 1:num_speakers
        
        ID_1 =  unique_indices_speakers(I_1);
        II_1_1 = indices_speakers(:,1) == ID_1;
        II_1_2 = indices_speakers(:,2) == ID_1;
    
        for I_2 = I_1:num_speakers
    
            count_speaker_pairs = count_speaker_pairs+1;
    
            ID_2 =  unique_indices_speakers(I_2);
            II_2_1 = indices_speakers(:,1) == ID_2;
            II_2_2 = indices_speakers(:,2) == ID_2;

            if ID_1 == ID_2
                II_pair_s_speakers(count_speaker_pairs) = true;

                II_test = II_1_1 & II_2_2;

                II_train = ~(II_1_1 | II_2_2);
            else
                II_test = (II_1_1 & II_2_2) | (II_1_2 & II_2_1);

                II_train = ~(II_1_1 | II_1_2 | II_2_1 | II_2_2);
            end

            II_cross_val(count_speaker_pairs,:) = {II_test, II_train};

            % update waitbar
            waitbar(count_speaker_pairs/num_speaker_pairs, h_wait);
        end
    
    end
    close(h_wait);

    % cross-validation using parallel processing
    cross_val_output_speakers = cell(num_speaker_pairs,1);
    sigma2_target_speakers = NaN(num_speaker_pairs,num_biG_methods);

    % initialize parallel processing
    poolobj = gcp('nocreate');
    if isempty(poolobj)
        numcores = feature('numcores');
        poolobj = parpool(numcores-1);
    end
    % initiate waitbar
    h_wait = waitbar(0,'Calibrating speaker scores');
    h_wait.UserData = [0 num_speaker_pairs];
    par_DataQueue = parallel.pool.DataQueue;
    afterEach(par_DataQueue, @(varargin) increment_waitbar(h_wait));

    parfor I_cross_val = 1:num_speaker_pairs

        II_test = II_cross_val{I_cross_val,1};
        scores_to_cal_speakers = scores_speakers(II_test);

        II_train = II_cross_val{I_cross_val,2};
        scores_s_speakers_train = scores_speakers(II_s_speakers & II_train);
        scores_d_speakers_train = scores_speakers(II_d_speakers & II_train);

        % LogReg regularization ceofficients
        if II_pair_s_speakers(I_cross_val)
            logreg_regularization_coefs = [kappa, num_speakers-1];
        else
            logreg_regularization_coefs = [kappa, num_speakers-2];
        end

        % biGauss calibration (EER, LogReg, KDE methods) + LogReg calibration
        [lnLR_speakers_temp, sigma2_target_speakers_temp] = ...
            biGaussianized_calibration_all(scores_to_cal_speakers, scores_s_speakers_train, scores_d_speakers_train, methods, ...
            Cllr_to_sigma2_coefs, logreg_regularization_coefs);

        % PAV calibration
        lnLR_speakers_PAV = pav_transform(scores_to_cal_speakers, scores_s_speakers_train, scores_d_speakers_train)';

        % output results [biG_EER, biG_LogReg, biG_KDE, LogReg, PAV]
        cross_val_output_speakers(I_cross_val) = {[lnLR_speakers_temp, lnLR_speakers_PAV]};
        sigma2_target_speakers(I_cross_val,:) = sigma2_target_speakers_temp;

        send(par_DataQueue, I_cross_val);
    end
    close(h_wait);

    % unpack outputs from parallel processing
    lnLR_speakers = NaN(num_methods,num_scores_speakers);
    h_wait = waitbar(count_speaker_pairs,'Unpacking cross-validation results');
    for I_cross_val = 1:num_speaker_pairs
        II_test = II_cross_val{I_cross_val,1}; 
        lnLR_speakers(:,II_test) = cross_val_output_speakers{I_cross_val}';
        waitbar(I_cross_val/num_speaker_pairs, h_wait);
    end
    close(h_wait);

    % target sigma
    sigma_target_speakers = sqrt(sigma2_target_speakers);
    mean_sigma_target_speakers = mean(sigma_target_speakers,1);
    
    fprintf('\n\nspeaker data\n')
    fprintf('\nsigma\nbiG_EER\tbiG_LogReg\tbiG_KDE\n')
    fprintf('%0.2f\t%0.2f\t%0.2f\n', ...
        mean_sigma_target_speakers)
    
    % Cllr
    lnLR_s_speakers = lnLR_speakers(:,II_s_speakers);
    lnLR_d_speakers = lnLR_speakers(:,II_d_speakers);
    
    Cllr_speakers = cllr(lnLR_s_speakers', lnLR_d_speakers');
    
    fprintf('\nCllr\t(top row train, bottom row test)')
    fprintf('\nbiG_EER\tbiG_LogReg\tbiG_KDE\tLogReg\tPAV\n')
    fprintf('%0.3f\t%0.3f\t%0.3f\t%0.3f\t%0.3f\n', ...
        Cllr_speakers)
    
    % plot mapping functions
    figure;
    scores_speakers_sorted = sort(scores_speakers)';
    plot(minmax_scores_speakers', [0; 0], '-k')
    hold on
    % PAV
    lnLR_pav_speakers_sorted = sort(lnLR_speakers(I_PAV,:));
    h1 = plot(scores_speakers_sorted, lnLR_pav_speakers_sorted, '-.m', 'LineWidth',plot_line_width);
    % logreg
    lnLR_minmax_logreg_speakers = minmax(lnLR_speakers(I_LogReg,:))';
    h2 = plot(minmax_scores_speakers, lnLR_minmax_logreg_speakers, '--r', 'LineWidth',plot_line_width);
    % biGaus
    lnLR_biGaus_speakers_sorted = sort(lnLR_speakers(I_biG_plot,:));
    h3 = plot(scores_speakers_sorted, lnLR_biGaus_speakers_sorted, ':b', 'LineWidth',plot_line_width);
    xlabel('Score', 'FontSize',plot_font_size);
    ylabel('ln(\Lambda)', 'FontSize',plot_font_size);
    axis tight
    grid on
    title(['speakers: \sigma target = ', num2str(mean_sigma_target_speakers(I_biG_plot), '%0.2f')]);
    legend([h1,h2,h3],{'PAV', 'LogReg', 'biGauss'}, 'Location','northwest')
    
    % plot pdf of lnLRs of speaker data
    minmax_x_speakers = [lnLR_biGaus_speakers_sorted(1); lnLR_biGaus_speakers_sorted(end)];
    step_plot_speakers = (minmax_x_speakers(2)-minmax_x_speakers(1))/399;
    x_speakers = minmax_x_speakers(1) : step_plot_speakers : minmax_x_speakers(2);
    
    half_sigma2_target_speakers = (mean_sigma_target_speakers(I_biG_plot)^2)/2;
    
    y_d_target_speakers = pdf('Normal', x_speakers, -half_sigma2_target_speakers, mean_sigma_target_speakers(I_biG_plot));
    y_s_target_speakers = pdf('Normal', x_speakers, half_sigma2_target_speakers, mean_sigma_target_speakers(I_biG_plot));
    
    kde_lnLR_d_biGaus_speakers = fitdist(lnLR_d_speakers(I_biG_plot,:)', 'Kernel');
    kde_lnLR_s_biGaus_speakers = fitdist(lnLR_s_speakers(I_biG_plot,:)', 'Kernel');
    
    y_d_biGaus_speakers = pdf(kde_lnLR_d_biGaus_speakers, x_speakers);
    y_s_biGaus_speakers = pdf(kde_lnLR_s_biGaus_speakers, x_speakers);
    
    kde_lnLR_d_logreg_speakers = fitdist(lnLR_d_speakers(I_LogReg,:)', 'Kernel');
    kde_lnLR_s_logreg_speakers = fitdist(lnLR_s_speakers(I_LogReg,:)', 'Kernel');
    
    y_d_logreg_speakers = pdf(kde_lnLR_d_logreg_speakers, x_speakers);
    y_s_logreg_speakers = pdf(kde_lnLR_s_logreg_speakers, x_speakers);
    
    figure;
    h1 = plot(x_speakers, y_d_target_speakers, '-r', 'LineWidth',plot_line_width);
    hold on
    plot(x_speakers, y_s_target_speakers, '-b', 'LineWidth',plot_line_width);
    h2 = plot(x_speakers, y_d_logreg_speakers, '--r', 'LineWidth',plot_line_width);
    plot(x_speakers, y_s_logreg_speakers, '--b', 'LineWidth',plot_line_width);
    h3 = plot(x_speakers, y_d_biGaus_speakers, ':r', 'LineWidth',plot_line_width);
    plot(x_speakers, y_s_biGaus_speakers, ':b', 'LineWidth',plot_line_width);
    xlim(minmax_x_speakers)
    lim_y = ylim;
    h4 = plot([0 0], lim_y, '-k');
    ylim(lim_y);
    uistack(h4,'bottom')
    grid on
    xlabel('ln(\Lambda)', 'FontSize',plot_font_size);
    ylabel('probability density', 'FontSize',plot_font_size);
    legend([h1,h2,h3],{'target', 'LogReg', 'biGauss'}, 'Location','northwest')
    title(['speakers: \sigma target = ', num2str(mean_sigma_target_speakers(I_biG_plot), '%0.2f')]);
    
    % Tippett plot
    % num_s_speakers = length(lnLR_s_speakers(I_biG_plot,:));
    % cum_prop_s_target_speakers = (1:num_s_speakers)/(num_s_speakers+1);
    % lnLR_s_target_speakers = icdf('Normal', cum_prop_s_target_speakers, half_sigma2_target_speakers, mean_sigma_target_speakers(I_biG_plot))';
    % num_d_speakers = length(lnLR_d_speakers(I_biG_plot,:));
    % cum_prop_d_target_speakers = (1:num_d_speakers)/(num_d_speakers+1);
    % lnLR_d_target_speakers = icdf('Normal', cum_prop_d_target_speakers, -half_sigma2_target_speakers, mean_sigma_target_speakers(I_biG_plot))';
    % 
    % plot_tippett(exp(lnLR_s_target_speakers), [], exp(lnLR_d_target_speakers), [], [], true, '-', true, [], [], true);

    plot_tippett(exp(lnLR_s_speakers(I_LogReg,:)), [], exp(lnLR_d_speakers(I_LogReg,:)), [], [], true, '-', true, [], [], true, mean_sigma_target_speakers(I_biG_plot));
    plot_tippett(exp(lnLR_s_speakers(I_LogReg,:)), [], exp(lnLR_d_speakers(I_LogReg,:)), [], [], false, '--', false, [], [], true);
    plot_tippett(exp(lnLR_s_speakers(I_biG_plot,:)), [], exp(lnLR_d_speakers(I_biG_plot,:)), [], [], false, ':', false, [], [], true);
    box on
    title(['speakers: \sigma target = ', num2str(mean_sigma_target_speakers(I_biG_plot), '%0.2f')]);
    
    xlim([-8 8])
    
    h_cf = gcf;
    h1 = findobj(h_cf, 'Color','red', 'LineStyle','-');
    h2 = findobj(h_cf, 'Color','red', 'LineStyle','--');
    h3 = findobj(h_cf, 'Color','red', 'LineStyle',':');
    legend([h1,h2, h3],{'target', 'LogReg', 'biGauss'}, 'Location','SouthWest')

else
    fprintf('\n\nTo run the cross-validated evaluation of speaker scores, in the options, set "evaluate_speakers" to "true".\nWARNING: Running the cross-validated evaluation of speaker scores takes a long time.\n')

end


%% Real data: glass

% glass scores from van Es et al (2017)
%   van Es A., Wiarda W., Hordijk M., Alberink I., Vergeer P. (2017). 
%       Implementation and assessment of a likelihood ratio approach for the evaluation of LA-ICP-MS evidence in forensic glass analysis.
%       Science & Justice, 57, 181–192.
%       http://dx.doi.org/10.1016/j.scijus.2017.03.002

score_file_glass = 'glass_scores.mat';
load(['./data/', score_file_glass], 'scores_s', 'scores_d', 'indices_s', 'indices_d');

% arrange data
scores_s_glass = scores_s';
indices_s_glass = indices_s';
scores_d_glass = scores_d';
indices_d_glass = indices_d';

num_s_glass = length(scores_s_glass);
num_d_glass = length(scores_d_glass);
num_glass = num_s_glass + num_d_glass;

scores_glass = [scores_d_glass, scores_s_glass];
indices_glass = [indices_d_glass, [indices_s_glass;indices_s_glass]];
II_s_glass = [false(num_d_glass,1); true(num_s_glass,1)];
II_d_glass = ~II_s_glass;

% data with -Inf scores removed
% to speed up processing, we will use these as test data, 
% then copy the lnLR corresponding to the smallest finite score as the lnLR for all the scores that had -Inf scores
II_d_negInf = scores_d_glass == -Inf;
scores_d_glass_noInf = scores_d_glass(~II_d_negInf);
indices_d_glass_noInf = indices_d_glass(:,~II_d_negInf);

scores_glass_noInf = [scores_d_glass_noInf, scores_s_glass];
indices_glass_noInf = [indices_d_glass_noInf, [indices_s_glass;indices_s_glass]];

num_d_glass_noInf = length(scores_d_glass_noInf);
num_glass_noInf = num_d_glass_noInf + num_s_glass;

num_d_glass_Inf = num_d_glass - num_d_glass_noInf;

% data with -Inf scores replaced by lowest finite score
% biGaussianized_calibration will handle -Inf values in training set, but we need these to plot score distributions
[min_finite_score_d, I_min_finite_score_d] = min(scores_d_glass_noInf);
scores_d_glass_finite = scores_d_glass;
scores_d_glass_finite(II_d_negInf) = min_finite_score_d;

scores_glass_finite = [scores_d_glass_finite, scores_s_glass];

% plot score distributions
minmax_scores_glass = minmax(scores_glass_finite);
step_scores_glass = (minmax_scores_glass(2) - minmax_scores_glass(1))/399;
x_scores_glass = minmax_scores_glass(1) : step_scores_glass : minmax_scores_glass(2);

kde_scores_s_glass = fitdist(scores_s_glass', 'Kernel');
kde_scores_d_glass = fitdist(scores_d_glass_finite', 'Kernel', 'Bandwidth', kde_scores_s_glass.Bandwidth);

y_s_scores_glass = pdf(kde_scores_s_glass, x_scores_glass);
y_d_scores_glass = pdf(kde_scores_d_glass, x_scores_glass);

figure;
plot(x_scores_glass, y_d_scores_glass, '-r', 'LineWidth',plot_line_width);
hold on
plot(x_scores_glass, y_s_scores_glass, '-b', 'LineWidth',plot_line_width);
xlim(minmax_scores_glass)
lim_y = ylim;
grid on
xlabel('scores', 'FontSize',plot_font_size);
ylabel('probability density', 'FontSize',plot_font_size);
title('glass: scores');

% run bi-Gaussian calibration (LogReg variant) without cross-validation in order to plot cdfs 
[~, sigma2_target_speakers_non_xval] = ...
    biGaussianized_calibration(0, scores_s_glass, scores_d_glass, ...
    "biGauss_LogReg", Cllr_to_sigma2_coefs, [kappa, num_s_glass], 1);
xlim(minmax_scores_glass)

if evaluate_glass
    fprintf('\n\nWARNING: Running the cross-validated evaluation of glass scores takes a very long time.\n')

    % leave-one-source-out leave-two-sources-out cross validation
    % 
    % each different-source pair is unique and there are no AvB and BvA pairs, so just go through each pair in turn
    %
    % this is slow
    %
    % to speed up processing, we use test data with -Inf scores removed, 
    % and copy the lnLR corresponding to the smallest finite score as the lnLR for all the scores that had -Inf scores

    num_methods = 5;
    num_biG_methods = 3;

    lnLR_glass = NaN(num_methods, num_glass_noInf);
    sigma2_target_glass = NaN(num_glass_noInf, num_biG_methods);
    
    % initialize parallel processing
    poolobj = gcp('nocreate');
    if isempty(poolobj)
        numcores = feature('numcores');
        poolobj = parpool(numcores-1);
    end
    % initiate waitbar
    h_wait = waitbar(0,'Calibrating glass scores');
    h_wait.UserData = [0 num_glass_noInf];
    par_DataQueue = parallel.pool.DataQueue;
    afterEach(par_DataQueue, @(varargin) increment_waitbar(h_wait));

    parfor I_glass = 1:num_glass_noInf
        % score to calibrate
        score_test = scores_glass_noInf(I_glass);
        % scores for training
        IDs_temp =  indices_glass(:,I_glass);
        II_hold_out = ...
            indices_glass(1,:) == IDs_temp(1) |...
            indices_glass(1,:) == IDs_temp(2) |...
            indices_glass(2,:) == IDs_temp(1) |...
            indices_glass(2,:) == IDs_temp(2);
        scores_train = scores_glass(~II_hold_out);
        II_s_train_glass = II_s_glass(~II_hold_out);
        scores_s_train_glass = scores_train(II_s_train_glass);
        scores_d_train_glass = scores_train(~II_s_train_glass);
        % biGauss and logreg calibrate
        if IDs_temp(1) == IDs_temp(2)
            logreg_regularization_coefs = [kappa, num_s_glass-1];
        else
            logreg_regularization_coefs = [kappa, num_s_glass-2];
        end

        % biGauss calibration (EER, LogReg, KDE methods) + LogReg calibration
        [lnLR_glass_temp, sigma2_target_glass_temp] = ...
            biGaussianized_calibration(score_test, scores_s_train_glass, scores_d_train_glass, methods, ...
            Cllr_to_sigma2_coefs, logreg_regularization_coefs);
        
        % PAV
        lnLR_glass_pav = pav_transform(score_test, scores_s_train_glass, scores_d_train_glass)'; % PAV can be trained on -Infs

        % output results [biG_EER, biG_LogReg, biG_KDE, LogReg, PAV]
        lnLR_glass(:,I_glass) = [lnLR_glass_temp, lnLR_glass_pav]';
        sigma2_target_glass(I_glass,:) = sigma2_target_glass_temp;
    
        % update waitbar
        send(par_DataQueue, I_glass);
    end
    close(h_wait);
    
    % target sigma
    sigma_target_glass = sqrt(sigma2_target_glass);
    mean_sigma_target_glass = mean(sigma_target_glass,1);

    fprintf('\n\nglass data\n')
    fprintf('\nsigma\nbiG_EER\tbiG_LogReg\tbiG_KDE\n')
    fprintf('%0.2f\t%0.2f\t%0.2f\n', ...
        mean_sigma_target_glass)
    
    % Cllr
    lnLR_s_glass = lnLR_glass(:,num_d_glass_noInf+1:end);
    lnLR_d_glass_no_Inf = lnLR_glass(:,1:num_d_glass_noInf);

    lnLR_d_glass = NaN(num_methods, num_d_glass);
    lnLR_d_glass(:,~II_d_negInf) = lnLR_d_glass_no_Inf;
    % lnLR_d_glass(:,II_d_negInf) = repmat(lnLR_d_glass_no_Inf(:,I_min_finite_score_d), [1,num_d_glass_Inf]);
    lnLR_d_glass(:,II_d_negInf) = repmat(min(lnLR_d_glass_no_Inf,[],2), [1,num_d_glass_Inf]);
    
    lnLR_glass_negInf = [lnLR_d_glass, lnLR_s_glass];

    Cllr_glass = cllr(lnLR_s_glass', lnLR_d_glass');
    
    fprintf('\nCllr\t(top row train, bottom row test)')
    fprintf('\nbiG_EER\tbiG_LogReg\tbiG_KDE\tLogReg\tPAV\n')
    fprintf('%0.3f\t%0.3f\t%0.3f\t%0.3f\t%0.3f\n', ...
        Cllr_glass)

    % plot mapping functions
    figure;
    scores_glass_sorted = sort(scores_glass_finite);
    plot(minmax_scores_glass', [0; 0], '-k')
    hold on
    % PAV
    lnLR_pav_glass_sorted = sort(lnLR_glass_negInf(I_PAV,:));
    h1 = plot(scores_glass_sorted, lnLR_pav_glass_sorted, '-.m', 'LineWidth',plot_line_width);
    % logreg
    lnLR_minmax_logreg_glass = minmax(lnLR_glass(I_LogReg,:))';
    h2 = plot(minmax_scores_glass', lnLR_minmax_logreg_glass, '--r', 'LineWidth',plot_line_width);
    % biGaus
    lnLR_biGaus_glass_sorted = sort(lnLR_glass_negInf(I_biG_plot,:));
    h3 = plot(scores_glass_sorted, lnLR_biGaus_glass_sorted, ':b', 'LineWidth',plot_line_width);
    xlabel('Score', 'FontSize',plot_font_size);
    ylabel('ln(\Lambda)', 'FontSize',plot_font_size);
    axis tight
    grid on
    title(['glass: \sigma target = ', num2str(mean_sigma_target_glass(I_biG_plot), '%0.2f')]);
    legend([h1,h2,h3],{'PAV', 'LogReg', 'biGauss'}, 'Location','southeast')
    
    % plot pdf of lnLRs of glass data
    % minmax_x_glass = [lnLR_biGaus_glass_sorted(1), lnLR_biGaus_glass_sorted(end)];
    minmax_x_glass = [-lnLR_biGaus_glass_sorted(end), lnLR_biGaus_glass_sorted(end)]; % to make plot symmetrical on target and show lnLR_d distribution
    step_plot_glass = (minmax_x_glass(2)-minmax_x_glass(1))/399;
    x_glass = minmax_x_glass(1) : step_plot_glass : minmax_x_glass(2);

    half_sigma2_target_glass = (mean_sigma_target_glass(I_biG_plot)^2)/2;
    
    y_d_target_glass = pdf('Normal', x_glass, -half_sigma2_target_glass, mean_sigma_target_glass(I_biG_plot));
    y_s_target_glass = pdf('Normal', x_glass, half_sigma2_target_glass, mean_sigma_target_glass(I_biG_plot));
    
    kde_lnLR_s_biGaus_glass = fitdist(lnLR_s_glass(I_biG_plot,:)', 'Kernel');
    kde_lnLR_d_biGaus_glass = fitdist(lnLR_d_glass(I_biG_plot,:)', 'Kernel', 'Bandwidth',0.1);
    
    y_d_biGaus_glass = pdf(kde_lnLR_d_biGaus_glass, x_glass);
    y_s_biGaus_glass = pdf(kde_lnLR_s_biGaus_glass, x_glass);
    
    kde_lnLR_s_logreg_glass = fitdist(lnLR_s_glass(I_LogReg,:)', 'Kernel');
    kde_lnLR_d_logreg_glass = fitdist(lnLR_d_glass(I_LogReg,:)', 'Kernel');
    
    y_d_logreg_glass = pdf(kde_lnLR_d_logreg_glass, x_glass);
    y_s_logreg_glass = pdf(kde_lnLR_s_logreg_glass, x_glass);
    
    figure;
    h1 = plot(x_glass, y_d_target_glass, '-r', 'LineWidth',plot_line_width);
    hold on
    plot(x_glass, y_s_target_glass, '-b', 'LineWidth',plot_line_width);
    h2 = plot(x_glass, y_d_logreg_glass, '--r', 'LineWidth',plot_line_width);
    plot(x_glass, y_s_logreg_glass, '--b', 'LineWidth',plot_line_width);
    h3 = plot(x_glass, y_d_biGaus_glass, ':r', 'LineWidth',plot_line_width);
    plot(x_glass, y_s_biGaus_glass, ':b', 'LineWidth',plot_line_width);
    xlim(minmax_x_glass)
    lim_y = ylim;
    h4 = plot([0 0], lim_y, '-k');
    ylim(lim_y);
    uistack(h4,'bottom')
    grid on
    xlabel('ln(\Lambda)', 'FontSize',plot_font_size);
    ylabel('probability density', 'FontSize',plot_font_size);
    legend([h1,h2,h3],{'target', 'LogReg', 'biGauss'}, 'Location','northeast')
    title(['glass: \sigma target = ', num2str(mean_sigma_target_glass(I_biG_plot), '%0.2f')]);

    % set(gca,'YLim',[0 0.5])
    
    % Tippett plot
    % cum_prop_s_target_glass = (1:num_s_glass)/(num_s_glass+1);
    % lnLR_s_target_glass = icdf('Normal', cum_prop_s_target_glass, half_sigma2_target_glass, mean_sigma_target_glass(I_biG_plot))';
    % cum_prop_d_target_glass = (1:num_d_glass)/(num_d_glass+1);
    % lnLR_d_target_glass = icdf('Normal', cum_prop_d_target_glass, -half_sigma2_target_glass, mean_sigma_target_glass(I_biG_plot))';
    % 
    % plot_tippett(exp(lnLR_s_target_glass), [], exp(lnLR_d_target_glass), [], [], true, '-', true, [], [], true);

    plot_tippett(exp(lnLR_s_glass(I_LogReg,:)), [], exp(lnLR_d_glass(I_LogReg,:)), [], [], true, '-', true, [], [], true, mean_sigma_target_glass(I_biG_plot));
    plot_tippett(exp(lnLR_s_glass(I_LogReg,:)), [], exp(lnLR_d_glass(I_LogReg,:)), [], [], false, '--', false, [], [], true);
    plot_tippett(exp(lnLR_s_glass(I_biG_plot,:)), [], exp(lnLR_d_glass(I_biG_plot,:)), [], [], false, ':', false, [], [], true);
    box on
    title(['glass: \sigma target = ', num2str(mean_sigma_target_glass(I_biG_plot), '%0.2f')]);
    
    xlim([-15 15])
    xticks(-15:3:15)
    
    h_cf = gcf;
    h1 = findobj(h_cf, 'Color','red', 'LineStyle','-');
    h2 = findobj(h_cf, 'Color','red', 'LineStyle','--');
    h3 = findobj(h_cf, 'Color','red', 'LineStyle',':');
    legend([h1,h2, h3],{'target', 'LogReg', 'biGauss'}, 'Location','SouthEast')
    
else
    fprintf('\n\nTo run the cross-validated evaluation of glass scores, in the options, set "evaluate_glass" to "true".\nWARNING: Running the cross-validated evaluation of glass scores takes a very long time.\n')

end


%% compare Cllr variability due to sampling variability 

if compare_Cllr_variability

    logreg_regularization_coefs = [kappa, num_s_sample];
    num_methods = 5;
    Cllr_variability = NaN(num_samples_Cllr_variability,num_methods);

    % single set of test data
    num_test = 10000;
    rng(1);
    x_s_test = randn([1 num_test]) * sigma_true + mu_s_true;
    x_d_test = randn([1 num_test]) * sigma_true + mu_d_true;
    x_test = [x_d_test, x_s_test];

    Cllr_true = sigma2_to_Cllr(sigma_true^2);
    
    % initialize parallel processing
    poolobj = gcp('nocreate');
    if isempty(poolobj)
        numcores = feature('numcores');
        poolobj = parpool(numcores-1);
    end
    % initiate random stream to replicate results when using parfor
    sc = parallel.pool.Constant(RandStream('threefry4x64_20'));
    % initiate waitbar
    h_wait = waitbar(0,'Comparing \itC\rm_{llr} variability');
    h_wait.UserData = [0 num_samples_Cllr_variability];
    par_DataQueue = parallel.pool.DataQueue;
    afterEach(par_DataQueue, @(varargin) increment_waitbar(h_wait));

    parfor I_sample = 1:num_samples_Cllr_variability
        % assign random stream to replicate results when using parfor
        stream = sc.Value;
        stream.Substream = I_sample;

        % generate Monte Carlo samples
        x_s_train = randn(stream, [1 num_s_sample]) * sigma_true + mu_s_true;
        x_d_train = randn(stream, [1 num_d_sample]) * sigma_true + mu_d_true;

        % bi-Gaussianized calibration
        lnLR_biGauss = ...
        biGaussianized_calibration ...
            (x_test, x_s_train, x_d_train, methods, ...
             Cllr_to_sigma2_coefs, logreg_regularization_coefs);

        % PAV calibration
        lnLR_PAV = pav_transform(x_test, x_s_train, x_d_train)';

        % combine
        lnLR = [lnLR_biGauss, lnLR_PAV];

        % Cllr
        Cllr_variability(I_sample,:) = cllr(lnLR(num_test+1:end,:), lnLR(1:num_test,:));

        % update waitbar
        send(par_DataQueue, I_sample);
    end
    close(h_wait);

    % violin plots
    h_fig = figure;
    % labels = {'biGauss(EER)', 'biGauss(LogReg)', 'biGauss(KDE)', 'LogReg', 'PAV'};
    labels = {'(a)', '(b)', '(c)', '(d)', '(e)'};
    violin(Cllr_variability, 'xlabel',labels, 'facecolor',[0.9 0.9 0.9], 'mc',[], 'plotlegend',false, 'same_area',true);
    ax=gca;
    ax.YLabel.String = '\itC\rm_{llr}';
    ax.YLabel.Rotation = 0;
    hold on
    lim_x = get(ax,'XLim');
    plot(lim_x,[Cllr_true;Cllr_true], '-k', 'LineWidth',1)

    % make figure wider
    fig_position = h_fig.Position;
    fig_position(3) = fig_position(3)*1.5;
    h_fig.Position = fig_position;

    % lim_y = get(ax,'YLim');
    % set(ax,'YLim', [0.24 0.3])
    % set(ax,'YLim', [0.24 0.36])
    % set(ax,'YLim', [0 1])
    
end



%% cleanup
rmpath('./functions/');