function plot_biGaussian(sigma, LR, x_tick_step, y_tick_step)
%
% display a perfectly-calibrated bi-Gaussian system and a likelihood ratio
%
% REQUIRED INPUTS:
%   sigma:          scalar  standard deviation of perfectly-calibrated Gaussian distributions (scaled as lnLR)
%   LR:             scalar  likelihood ratio to display
% OPTIONAL INPUTS:
%   x_tick_step:    scalar  base for logaritmic scale of x axis, e.g., 10 or 2
%   y_tick_step:    scalar  size of steps for scale of y axis as a multiplier of the lowest of the two likelihoods of the likelihood ratio,
%                           e.g., enter "1" to show every integer multiple of the lowest likelihood

% natural logarithm of likelihood ratio
lnLR = log(LR);

% parameters for perfectly-calibrated bi-Gaussian system (scaled as lnLR)
sigma2 = sigma^2;
mu_s = sigma2/2;
mu_d = -mu_s;

% x (lnLR) plotting range
x_plot_max = mu_s + 3*sigma;
if abs(lnLR) > x_plot_max
    x_plot_max = abs(lnLR)*1.1;
end
x = (-x_plot_max : 2*x_plot_max/399 : x_plot_max);

% y (pdf) for different-source and same-source distributions
y_s = normpdf(x, mu_s, sigma);
y_d = normpdf(x, mu_d, sigma);

% create figure and adjust aspect ratio
h = figure;
fig_pos = h.Position;
fig_pos(3) = fig_pos(3)*1.5;
h.Position = fig_pos;

% plot different-source and same-source distributions
plot_line_width = 1;
plot_font_size = 12;
plot(x, y_d, '-r', 'LineWidth',plot_line_width);
hold on
plot(x, y_s, '-b', 'LineWidth',plot_line_width);
xlim([x(1) x(end)]);
xlabel('likelihood ratio', 'FontSize',plot_font_size);
ylabel('relative likelihood', 'FontSize',plot_font_size);

lim_y = ylim;
h_zero_line = plot([0 0], lim_y, '-k');
ylim(lim_y);
uistack(h_zero_line,'bottom')
grid on

% write scale on x axis
if nargin < 3 || isempty(x_tick_step)
    if x_plot_max > log(1024)
        base = 10;
    else
        base = 2;
    end
else
    base = x_tick_step;
end

lim_x_tick = floor(x(end)/log(base));
x_scale_log_base = -lim_x_tick : lim_x_tick;
x_scale_ln = x_scale_log_base*log(base);
x_scale_linear = base.^x_scale_log_base;
x_scale_str = num2str_fractions_and_commas(x_scale_linear);

xticks(x_scale_ln);
xticklabels(x_scale_str);
xtickangle(90)

% y (pdf) for likelihood ratio
likelihood_s = normpdf(lnLR, mu_s, sigma);
likelihood_d = normpdf(lnLR, mu_d, sigma);

% plot likelihood ratio
plot([lnLR lnLR], [0 max([likelihood_s likelihood_d])],  '-g', 'LineWidth',plot_line_width);
plot([lnLR lnLR], [likelihood_s likelihood_d],  'og', 'LineWidth',plot_line_width);
plot([x(1) lnLR], [likelihood_s likelihood_s],  '--g', 'LineWidth',plot_line_width);
plot([x(1) lnLR], [likelihood_d likelihood_d],  '--g', 'LineWidth',plot_line_width);

% write scale on y axis
likelihood_min = min([likelihood_s likelihood_d]);
% rewrite values of exiting ticks
y_ticks_default = yticks;
y_scale_rel = y_ticks_default / likelihood_min;
% use y_tick_step if provided
if nargin>3 && ~isempty(y_tick_step)
    lim_y_rel = lim_y(2)/likelihood_min;
    num_y_ticks = lim_y_rel / y_tick_step;
    if num_y_ticks > 30
        warning('When calling plot_Gaussian, the y_tick_step value would result in more than 30 ticks. Please use a larger y_tick_step value. The plot created has used a default y_tick_step value.')
    else
        y_step_rel = lim_y_rel/num_y_ticks;
        y_scale_rel = 0 : y_step_rel : lim_y_rel;
        if y_tick_step>=1
            y_scale_rel = round(y_scale_rel);
        end
        y_scale_density = y_scale_rel * likelihood_min;
        yticks(y_scale_density);
    end
end
yticklabels(y_scale_rel);
