#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
2025-07-10 Phil Weber
    Utility functions in support of BiGaussianized calibration.
"""

import numpy as np
from scipy.stats import norm

def cllr_to_sigma2(cllr, cllr_to_sigma2_coefs):
    """ Return the mapped value of sigma^2 for the provided Cllr and coefficients.

    Used for bi-Gaussian calibration (Refer to Morrison, 2023).

    :param cllr: Cllr to map
    :param cllr_to_sigma2_coefs: Regression coefficients for mapping sigma2 to ln(LR)
    """
    Cllr_to_sigma2_coef_b, Cllr_to_sigma2_coef_c = cllr_to_sigma2_coefs
    return -np.log((np.log(cllr) / Cllr_to_sigma2_coef_b) + 1) / Cllr_to_sigma2_coef_c

def cllr(lnLRs_s, lnLRs_d) -> float:
    """
    Calculate log likelihood ratio cost (Cllr)

    Args:
        lnLRs_s: natural log likelihood ratios obtained from same-source comparisons
        lnLRs_d: natural log likelihood ratios obtained from different-source comparisons
    Returns:
        log likelihood ratio cost (Cllr)
    """

    # cllr_val = (np.mean(np.log2(1 + np.exp(-lnLRs_s))) + np.mean(np.log2(1 + np.exp(lnLRs_d)))) / 2

    cllr_s, cllr_d = 0, 0
    if len(lnLRs_s) > 0:
        cllr_s = np.mean(np.log2(1 + np.exp(-lnLRs_s)))
        # cllr_s = np.mean(-lnLRs_s / np.log(2))
    if len(lnLRs_d) > 0:
        cllr_d = np.mean(np.log2(1 + np.exp(lnLRs_d)))
        # cllr_d = np.mean(lnLRs_d / np.log(2))

    cllr_val = (cllr_s + cllr_d) / 2
    return cllr_val

def mix_norm_cdf(sample=0, weights=[0], means=[0], covars=[1]):
    """Return cdf value for given sample and distribution.
    
    Return the cdf for the given value under the Gaussian Mixture Model with the given parameters.
    """
    mcdf = 0.0
    weights = np.asarray(weights)
    means = np.asarray(means)
    stdevs = np.sqrt(np.asarray(covars))
    for ii in range(len(weights)):
        mcdf += weights[ii] * norm.cdf(sample, loc=means[ii], scale=stdevs[ii])
        
    return mcdf
    
def np_unique_last(AA: np.ndarray):
    """Return the unique elements in array, with the indices of their last occurrences.

    Equivalent to Matlab function unique(AA, 'last')

    :param AA: Array to process
    """
    _, ii = np.unique(AA[::-1], return_index=True)
    ii = (len(AA) - 1 - ii)
    return AA[ii], ii
